/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.fritzapp.service;

import de.avm.android.fritzapp.R;
import de.avm.android.fritzapp.com.ComSettingsChecker;
import de.avm.android.fritzapp.com.discovery.BoxInfo;
import de.avm.android.fritzapp.gui.FRITZApp;
import de.avm.android.fritzapp.util.ResourceHelper;
import android.app.Notification;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.text.TextUtils;

/**
 * Manager for connection status notifications
 */
public class ComStatus
{
	public static final int ID = 1;

	private Service mService;
	private Notification mNotification = null;

	// last notified
	private String mUdn = null;
	private boolean mHasError = false;
	private int mSipState = ComSettingsChecker.SIP_NOTREGISTERED;

	public ComStatus(Service service)
	{
		mService = service;
	}
	
	/**
	 * Referesh notification
	 * 
	 * @return true if there where changes
	 */
	public boolean refresh()
	{
		BoxInfo boxInfo = ComSettingsChecker.getBoxInfo();
		String udn = (boxInfo == null) ? null : boxInfo.getUdn();
		boolean hasError = false;
		if (TextUtils.isEmpty(udn))
			hasError = ComSettingsChecker.getLastError().isError();
		int sipState = ComSettingsChecker.getSipState();
		if ((udn != mUdn) || (hasError != mHasError) || (sipState != mSipState))
		{
			mUdn = udn;
			mHasError = hasError;
			mSipState = sipState;
			update();
			return true;
		}
		return false;
	}
	
	/**
	 * Remove notification
	 */
	public void clear()
	{
		mUdn = null;
		mHasError = false;
		mSipState = ComSettingsChecker.SIP_NOTREGISTERED;
		stopForeground();
	}

	private void update()
	{
		if (!ComSettingsChecker.isConnected())
		{
			stopForeground();
			return;
		}

		if (mNotification == null)
		{
			mNotification = new Notification(R.drawable.state_connected,
					mService.getResources().getText(R.string.connection_on).toString(), 0);
			mNotification.flags |= Notification.FLAG_ONGOING_EVENT;
		}

		String message;
		switch(mSipState)
		{
			case ComSettingsChecker.SIP_AWAY:
				mNotification.icon = R.drawable.state_sipaway;
				mNotification.flags |= Notification.FLAG_SHOW_LIGHTS;
				mNotification.ledARGB = 0xffff0000; /* red */
				mNotification.ledOnMS = 125;
				mNotification.ledOffMS = 2875;
				message = mService.getResources().getText(R.string.regfailed).toString(); 
				break;
				
			case ComSettingsChecker.SIP_IDLE:
				mNotification.icon = R.drawable.state_sipidle;
				mNotification.flags &= ~Notification.FLAG_SHOW_LIGHTS;
				message = mService.getResources().getText(R.string.reg).toString();
				break;
				
			case ComSettingsChecker.SIP_AVAILABLE:
				mNotification.icon = R.drawable.state_sipavailable;
				mNotification.flags &= ~Notification.FLAG_SHOW_LIGHTS;
				message = mService.getResources().getText(R.string.regok).toString();
				break;

			case ComSettingsChecker.SIP_NOTREGISTERED:
			default:
				mNotification.icon = R.drawable.state_connected;
				mNotification.flags &= ~Notification.FLAG_SHOW_LIGHTS;
				message = mService.getResources().getText(R.string.regno).toString();
				break;
		}
		
		mNotification.setLatestEventInfo(mService,
				ResourceHelper.getApplicationName(mService),
				message, PendingIntent.getActivity(mService, 0, new Intent(mService,
						FRITZApp.class).addFlags(Intent.FLAG_ACTIVITY_NEW_TASK), 0));
		startForeground();
	}
	
	private void startForeground()
	{
		mService.startForeground(ID, mNotification);
	}
	
	private void stopForeground()
	{
		if (mNotification == null) return;

		mService.stopForeground(true);
		mNotification = null;
	}
}
