/*
 * Copyright (C) 2011 AVM GmbH <info@avm.de>
 * Copyright (C) 2009 The Sipdroid Open Source Project
 * Copyright (C) 2006 The Android Open Source Project
 * 
 * This file is part of Sipdroid (http://www.sipdroid.org)
 * 
 * Sipdroid is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package de.avm.android.fritzapp.sipua.phone;

import de.avm.android.fritzapp.R;
import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

/**
 * Button grid for dialpad
 *
 * @attr ref R.styleable#ButtonGridLayout_maxHorizontalSpacing
 * @attr ref R.styleable#ButtonGridLayout_maxVerticalSpacing
 * @attr ref R.styleable#ButtonGridLayout_columns
*/
public class ButtonGridLayout extends ViewGroup
{
	public final int IGNORE_MAX_SPACING = 0;
	private final int DEFAULT_NUMCOLUMNS = 3;

	private int mMaxHorizontalSpacing = IGNORE_MAX_SPACING;
	private int mMaxVerticalSpacing = IGNORE_MAX_SPACING;
	private int mNumColumns = DEFAULT_NUMCOLUMNS;
    
	public int getMaxHorizontalSpacing()
	{
		return mMaxHorizontalSpacing;
	}
    
	public int getMaxVerticalSpacing()
	{
		return mMaxVerticalSpacing;
	}
	
	public int getNumColumns()
	{
		return mNumColumns;
	}
	
    public ButtonGridLayout(Context context)
    {
        super(context);
    }

    public ButtonGridLayout(Context context, AttributeSet attrs)
    {
        super(context, attrs);
        initAttributes(attrs);
    }

    public ButtonGridLayout(Context context, AttributeSet attrs, int defStyle)
    {
        super(context, attrs, defStyle);
        initAttributes(attrs);
    }
    
    private void initAttributes(AttributeSet attrs)
    {
    	TypedArray array = getContext().obtainStyledAttributes(
    			attrs,R.styleable.ButtonGridLayout);
    	mMaxHorizontalSpacing = array.getDimensionPixelOffset(
    			R.styleable.ButtonGridLayout_maxHorizontalSpacing,
    			IGNORE_MAX_SPACING);
    	mMaxVerticalSpacing = array.getDimensionPixelOffset(
    			R.styleable.ButtonGridLayout_maxVerticalSpacing,
    			IGNORE_MAX_SPACING);
    	mNumColumns = array.getInt(R.styleable.ButtonGridLayout_numColumns,
    			DEFAULT_NUMCOLUMNS);
    	array.recycle();
    }
    
    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b)
    {
        final int rows = getNumRows();
        final View child0 = getChildAt(0);
        final int childWidth = child0.getMeasuredWidth();
        final int childHeight = child0.getMeasuredHeight();

        int extraMarginLeft = 0;
        int yGap = ((getHeight() - getPaddingTop() - getPaddingBottom()) -
        		rows * childHeight) / (rows - 1);
        if ((getMaxVerticalSpacing() > IGNORE_MAX_SPACING) &&
        		(yGap > getMaxVerticalSpacing()))
        	yGap = getMaxVerticalSpacing(); 
        int xGap = (getWidth() - getPaddingLeft() - getPaddingRight() -
        		getNumColumns() * childWidth) / (getNumColumns() - 1);
        if ((getMaxHorizontalSpacing() > IGNORE_MAX_SPACING) &&
        		(xGap > getMaxHorizontalSpacing()))
        {
        	xGap = getMaxHorizontalSpacing();
        	extraMarginLeft = (getWidth() - getPaddingLeft() - getPaddingRight() -
            		getNumColumns() * childWidth - (getNumColumns() - 1) * xGap) / 2;
        }

        for (int row = 0; row < rows; row++)
        {
        	int yPos = getPaddingTop() + (childHeight + yGap) * row;
            for (int col = 0; col < getNumColumns(); col++)
            {
                int cell = row * getNumColumns() + col;
                if (cell >= getChildCount()) break;
                
                View child = getChildAt(cell);
            	int xPos = getPaddingLeft() + extraMarginLeft +
            			(childWidth + xGap) * col;
                child.layout(xPos, yPos, xPos + childWidth, yPos + childHeight);
            }
        }
    }

    private int getNumRows()
    {
        return (getChildCount() + getNumColumns() - 1) / getNumColumns(); 
    }
    
    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = getPaddingLeft() + getPaddingRight();
        int height = getPaddingTop() + getPaddingBottom();
        
        // Measure the first child and get it's size
        View child = getChildAt(0);
        child.measure(MeasureSpec.UNSPECIFIED , MeasureSpec.UNSPECIFIED);
        int childWidth = child.getMeasuredWidth();
        int childHeight = child.getMeasuredHeight();
        // Make sure the other children are measured as well, to initialize
        for (int i = 1; i < getChildCount(); i++) {
            getChildAt(0).measure(MeasureSpec.UNSPECIFIED , MeasureSpec.UNSPECIFIED);
        }
        // All cells are going to be the size of the first child
        width += getNumColumns() * childWidth;
        height += getNumRows() * childHeight;
        
        width = resolveSize(width, widthMeasureSpec);
        height = resolveSize(height, heightMeasureSpec);
        setMeasuredDimension(width, height);
    }

}
