/*
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License ("License") as
 * published by the Free Software Foundation  (version 3 of the License).
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

public class StringSpec implements Parcelable
{
    protected int mMinChars;
    protected int mMaxChars;
    protected String mAllowedChars;

    public int getMinChars()
    {
        return mMinChars;
    }

    public int getMaxChars()
    {
        return mMaxChars;
    }

    public String getAllowedChars()
    {
        return mAllowedChars;
    }

    public StringSpec(int minChars, int maxChars, String allowedChars)
            throws IllegalArgumentException
    {
        if (minChars < 0)
            throw new IllegalArgumentException("Argument minChars must not be smaller than 0");
        if (maxChars < minChars)
            throw new IllegalArgumentException(
                    "Argument maxChars must not be smaller than minChars");
        if (TextUtils.isEmpty(allowedChars))
            throw new IllegalArgumentException("Argument allowedChars must not be empty or null");

        mMinChars = minChars;
        mMaxChars = maxChars;
        mAllowedChars = allowedChars;
    }

    /**
     * Validates value with the spec
     *
     * @param value text to validate
     * @return true if valid
     * @throws IllegalArgumentException invalid character in value
     */
    public boolean validate(CharSequence value)
            throws IllegalArgumentException
    {
        int len = (value == null) ? 0 : value.length();
        if ((len < mMinChars) || (len > mMaxChars))
            return false;

        for (int ii = 0; ii < len; ii++)
        {
            int code = Character.getNumericValue(value.charAt(ii));
            if (code < 0)
                throw new IllegalArgumentException("Invalid character value \"" +
                        value.subSequence(ii, ii + 1) + "\"");
            if (mAllowedChars.indexOf(code) < 0)
                return false;
        }

        return true;
    }

    public int describeContents()
    {
        return 0;
    }

    public static final Parcelable.Creator<StringSpec> CREATOR =
            new Parcelable.Creator<StringSpec>()
            {
                public StringSpec createFromParcel(Parcel in)
                {
                    return new StringSpec(in);
                }

                public StringSpec[] newArray(int size)
                {
                    return new StringSpec[size];
                }
            };

    public StringSpec(Parcel in)
    {
        mMinChars = in.readInt();
        mMaxChars = in.readInt();
        mAllowedChars = in.readString();
    }

    public void writeToParcel(Parcel out, int flags)
    {
        out.writeInt(mMinChars);
        out.writeInt(mMaxChars);
        out.writeString(mAllowedChars);
    }
}