/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap.voip;

import android.text.TextUtils;

import java.io.IOException;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.UnsignedNumber;
import de.avm.android.tr064.soap.ISoapCredentials;

public class SetClient extends VoIPConfSoapHelper<Boolean>
{
	private int mIndex = 0;
	private String mPassword;
	private String mName;
	private String mId = null;
	private String mOutgoingNumber;
	
	/**
     * @param index index of client to set
	 * @throws IllegalArgumentException
	 * 		illegal value of index 
	 */
	public SetClient(ISoapCredentials soapCredentials, int index,
			String password, String name, String outgoingNumber)
	{
		super(soapCredentials);

		UnsignedNumber.validateUi2(index, "index");
		
		mIndex = index;
		mPassword = password;
		mName = name;
		mOutgoingNumber = (outgoingNumber == null) ? "" : outgoingNumber;
	}

	public SetClient(ISoapCredentials soapCredentials, int index,
			String password, String name, String outgoingNumber, String id)
	{
		this(soapCredentials, index, password, name, outgoingNumber);
		mId = id;
	}

    public int getIndex()
    {
        return mIndex;
    }

    /**
     * @param index index of client to set
     * @throws IllegalArgumentException
     * 		illegal value of index
     */
    public SetClient setIndex(int index)
    {
        UnsignedNumber.validateUi2(index, "index");

        mIndex = index;
        return this;
    }

    public String getPassword()
    {
        return mPassword;
    }

    public SetClient setPassword(String password)
    {
        mPassword = password;
        return this;
    }

    public String getName()
    {
        return mName;
    }

    public SetClient setName(String name)
    {
        mName = name;
        return this;
    }

    public String getId()
    {
        return mId;
    }

    public SetClient setId(String id)
    {
        mId = id;
        return this;
    }

    public String getOutgoingNumber()
    {
        return mOutgoingNumber;
    }

    public SetClient setOutgoingNumber(String outgoingNumber)
    {
        mOutgoingNumber = (outgoingNumber == null) ? "" : outgoingNumber;
        return this;
    }

    @Override
	public Boolean getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
		// no out parameter, check for error response
		exceptOnFault(getSoapBody());
		return Boolean.TRUE;
	}

	@Override
	public String getSoapMethodParameter()
	{
		String params = "<NewX_AVM-DE_ClientIndex>" + Integer.toString(mIndex) + "</NewX_AVM-DE_ClientIndex>" +
				"<NewX_AVM-DE_ClientPassword>" + encodeEntities(mPassword) + "</NewX_AVM-DE_ClientPassword>" +
				"<NewX_AVM-DE_PhoneName>" + mName + "</NewX_AVM-DE_PhoneName>" +
				"<NewX_AVM-DE_OutGoingNumber>" + mOutgoingNumber + "</NewX_AVM-DE_OutGoingNumber>";
		if (!TextUtils.isEmpty(mId))
			params += "<NewX_AVM-DE_ClientId>" + encodeEntities(mId) + "</NewX_AVM-DE_ClientId>";
		return params;
	}

	@Override
	protected String filterSoapBeforeTrace(String body)
	{
        return replaceSecrets(body, new String[] { "NewX_AVM-DE_ClientPassword" });
	}
	
	@Override
	public String getSoapMethod()
	{
		if (TextUtils.isEmpty(mId)) return "X_AVM-DE_SetClient";
		return "X_AVM-DE_SetClient2";
	}
}