/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap.wlanconfiguration;

import java.io.IOException;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.UnsignedNumber;
import de.avm.android.tr064.model.WLANHybridMode;
import de.avm.android.tr064.soap.ISoapCredentials;

public class SetWLANHybridMode extends WLANSoapHelper<Boolean>
{
	private WLANHybridMode mMode;
	
	/**
	 * Sets WLAN hybrid mode info
	 * 
	 * @param interfaceIndex
	 * 		Index of WLAN configuration interface
	 * @param mode
	 * 		Mode to be set
	 * @throws IllegalArgumentException
	 * 		illegal value of a member of mode 
	 */
	public SetWLANHybridMode(ISoapCredentials soapCredentials, int interfaceIndex,
			WLANHybridMode mode)
	{
		super(soapCredentials, interfaceIndex);
		
		UnsignedNumber.validateUi4(mode.getMaxSpeedDS(), "MaxSpeedDS");
		UnsignedNumber.validateUi4(mode.getMaxSpeedUS(), "MaxSpeedUS");
		
		mMode = mode;
	}

	@Override
	public Boolean getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
		// no out parameter, check for error response
		exceptOnFault(getSoapBody());
		return Boolean.TRUE;
	}

	@Override
	public String getSoapMethodParameter()
	{
		return "<NewEnable>" + Integer.toString((mMode.isEnable()) ? 1 : 0) + "</NewEnable>" +
				"<NewBeaconType>" + mMode.getBeaconType().toString() + "</NewBeaconType>" +
				"<NewKeyPassphrase>" + encodeEntities(mMode.getKeyPassphrase()) + "</NewKeyPassphrase>" +
				"<NewSSID>" + encodeEntities(mMode.getSsid()) + "</NewSSID>" +
				"<NewBSSID>" + encodeEntities(mMode.getBssid()) + "</NewBSSID>" +
				"<NewTrafficMode>" + mMode.getTrafficMode().toString() + "</NewTrafficMode>" +
				"<NewManualSpeed>" + Integer.toString((mMode.isManualSpeed()) ? 1 : 0) + "</NewManualSpeed>" +
				"<NewMaxSpeedDS>" + Long.toString(mMode.getMaxSpeedDS()) + "</NewMaxSpeedDS>" +
				"<NewMaxSpeedUS>" + Long.toString(mMode.getMaxSpeedUS()) + "</NewMaxSpeedUS>";
	}

	@Override
	protected String filterSoapBeforeTrace(String body)
	{
        return replaceSecrets(body, new String[] { "NewKeyPassphrase" });
	}

	@Override
	public String getSoapMethod()
	{
		return "X_AVM-DE_SetWLANHybridMode";
	}
}
